<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistBroadcastsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistBroadcasts\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\TherapistBroadcastsImport;
use App\Models\Broadcast;
use App\Models\Therapist;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Str;

class TherapistBroadcastsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist.add_media');
        $this->middleware('permission:therapist.add_media', ['only' => ['store']]);
        $this->middleware('permission:therapist.add_media', ['only' => ['update']]);
        $this->middleware('permission:therapist.add_media', ['only' => ['destroy']]);
        $this->middleware('permission:therapist.add_media', ['only' => ['export']]);
        $this->middleware('permission:therapist.add_media', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist)
    {
        if ($req != null) {
            $therapist_broadcasts =  $therapist->therapist_broadcasts();
            if ($req->trash && $req->trash == 'with') {
                $therapist_broadcasts =  $therapist_broadcasts->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_broadcasts =  $therapist_broadcasts->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_broadcasts = $therapist_broadcasts->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_broadcasts = $therapist_broadcasts->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_broadcasts = $therapist_broadcasts->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_broadcasts = $therapist_broadcasts->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_broadcasts = $therapist_broadcasts->get();
                return $therapist_broadcasts;
            }
            $therapist_broadcasts = $therapist_broadcasts->get();
            return $therapist_broadcasts;
        }
        $therapist_broadcasts = $therapist->therapist_broadcasts()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_broadcasts;
    }


    /*********View All TherapistBroadcasts  ***********/
    public function index(Request $request, Therapist $therapist)
    {
        $therapist_broadcasts = $this->getter($request, null, $therapist);
        return view('super_admins.therapists.therapist_broadcasts.index', compact('therapist_broadcasts', 'therapist'));
    }

    /*********View Create Form of Broadcast  ***********/
    public function create(Therapist $therapist)
    {
        $tags = Tag::active()->get();
        return view('super_admins.therapists.therapist_broadcasts.create', compact('therapist', 'tags'));
    }

    /*********Store Broadcast  ***********/
    public function store(CreateRequest $request, Therapist $therapist)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if ($request->link_type == 'internal' && $request->file_type == 'audio') {
                $data['audio'] = uploadFile($request, 'file', 'therapist_broadcasts');
            } else {
                $data['video'] = uploadFile($request, 'file', 'therapist_broadcasts');
            }
            //$data['image'] = uploadCroppedFile($request,'image','therapist_broadcasts');
            $therapist_broadcast = $therapist->therapist_broadcasts()->create($data);
            $therapist_broadcast->slug = Str::slug($therapist_broadcast->name . ' ' . $therapist_broadcast->id, '-');
            $therapist_broadcast->save();
            $therapist_broadcast->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_broadcasts.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_broadcasts.index', $therapist->id)->with('message', 'Broadcast Created Successfully')->with('message_type', 'success');
    }

    /*********View Broadcast  ***********/
    public function show(Therapist $therapist, Broadcast $therapist_broadcast)
    {
        if ($therapist->id != $therapist_broadcast->therapist_id) {
            return redirect()->back()->with('message', 'Broadcast Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_broadcasts.show', compact('therapist_broadcast', 'therapist'));
    }

    /*********View Edit Form of Broadcast  ***********/
    public function edit(Therapist $therapist, Broadcast $therapist_broadcast)
    {
        if ($therapist->id != $therapist_broadcast->therapist_id) {
            return redirect()->back()->with('message', 'Broadcast Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_broadcasts.edit', compact('therapist_broadcast', 'therapist'));
    }

    /*********Update Broadcast  ***********/
    public function update(CreateRequest $request, Therapist $therapist, Broadcast $therapist_broadcast)
    {
        if ($therapist->id != $therapist_broadcast->therapist_id) {
            return redirect()->back()->with('message', 'Broadcast Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if ($request->link_type == 'internal' && $request->file_type == 'audio') {
                if ($request->file) {
                    $data['audio'] = uploadFile($request, 'file', 'therapist_broadcasts');
                } else {
                    $data['audio'] = $therapist_broadcast->audio;
                }
            } else {
                if ($request->file) {
                    $data['video'] = uploadFile($request, 'file', 'therapist_broadcasts');
                } else {
                    $data['video'] = $therapist_broadcast->video;
                }
            }
            $therapist_broadcast->update($data);
            $therapist_broadcast = Broadcast::find($therapist_broadcast->id);
            $slug = Str::slug($therapist_broadcast->name . ' ' . $therapist_broadcast->id, '-');
            $therapist_broadcast->update([
                'slug' => $slug
            ]);
            $therapist_broadcast->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_broadcasts.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_broadcasts.index', $therapist->id)->with('message', 'Broadcast Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_broadcasts = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_broadcasts." . $extension;
        return Excel::download(new TherapistBroadcastsExport($therapist_broadcasts), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistBroadcastsImport, $file);
        return redirect()->back()->with('message', 'Broadcast Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE Broadcast ***********/
    public function destroy(Therapist $therapist, Broadcast $therapist_broadcast)
    {
        if ($therapist->id != $therapist_broadcast->therapist_id) {
            return redirect()->back()->with('message', 'Broadcast Not Found')->with('message_type', 'error');
        }
        $therapist_broadcast->delete();
        return redirect()->back()->with('message', 'Broadcast Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE Broadcast ***********/
    public function destroyPermanently(Request $request, Therapist $therapist, $therapist_broadcast)
    {
        $therapist_broadcast = Broadcast::withTrashed()->find($therapist_broadcast);
        if ($therapist_broadcast) {
            if ($therapist_broadcast->trashed()) {
                if ($therapist_broadcast->image && file_exists(public_path($therapist_broadcast->image))) {
                    unlink(public_path($therapist_broadcast->image));
                }
                $therapist_broadcast->forceDelete();
                return redirect()->back()->with('message', 'Broadcast Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Broadcast is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Broadcast Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore Broadcast***********/
    public function restore(Request $request, Therapist $therapist, $therapist_broadcast)
    {
        $therapist_broadcast = Broadcast::withTrashed()->find($therapist_broadcast);
        if ($therapist_broadcast->trashed()) {
            $therapist_broadcast->restore();
            return redirect()->back()->with('message', 'Broadcast Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Broadcast Not Found')->with('message_type', 'error');
        }
    }
}
